
library ieee;
use ieee.std_logic_1164.all;


package procdefs is

	function log(num, base : integer) return integer;
	
	constant bit_wd   : integer := 8;
	constant num_regs : integer := 8;
	constant reg_wd   : integer := 3;
	
	constant nop : std_logic_vector(bit_wd - 1 downto 0) := "00000000";
	constant mal : std_logic_vector(bit_wd - 1 downto 0) := "00000001";
	constant mah : std_logic_vector(bit_wd - 1 downto 0) := "00000010";
	constant jmp : std_logic_vector(bit_wd - 1 downto 0) := "00000011";
	
	constant reg0 : std_logic_vector(reg_wd - 1 downto 0) := "000";
	constant reg1 : std_logic_vector(reg_wd - 1 downto 0) := "001";
	constant reg2 : std_logic_vector(reg_wd - 1 downto 0) := "010";
	constant reg3 : std_logic_vector(reg_wd - 1 downto 0) := "011";
	constant reg4 : std_logic_vector(reg_wd - 1 downto 0) := "100";
	constant reg5 : std_logic_vector(reg_wd - 1 downto 0) := "101";
	constant reg6 : std_logic_vector(reg_wd - 1 downto 0) := "110";
	constant reg7 : std_logic_vector(reg_wd - 1 downto 0) := "111";
	
	constant lui_base : std_logic_vector(bit_wd/2 - 1 downto 0) := "0001";
	constant ldu_base : std_logic_vector(bit_wd/2 - 1 downto 0) := "0010";
	constant lds_base : std_logic_vector(bit_wd/2 - 1 downto 0) := "0011";
	constant acc_base : std_logic_vector(bit_wd/2 - 1 downto 0) := "0100";
	constant add_base : std_logic_vector(bit_wd/2 - 1 downto 0) := "0101";
	constant ori_base : std_logic_vector(bit_wd/2 - 1 downto 0) := "0110";
	constant adi_base : std_logic_vector(bit_wd/2 - 1 downto 0) := "0111";
	
	function lui(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector;
	function ldu(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector;
	function lds(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector;
		
	function rac(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector;
	function wac(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector;
	
	function add(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector;
	function sub(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector;
		
	function addi(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector;
	function ori(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector;
	
end package procdefs;

package body procdefs is

	function log(num, base : integer) return integer is
		variable exp : integer := 0;
		variable tmp : integer := num;
	begin
		while (tmp > 1) loop
			tmp := tmp / base;
			exp := exp + 1;
		end loop;

		return exp;

	end function;
	
	
	function lui(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector is
	begin
		return lui_base & imm; 
	end;
		
		
	function ldu(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector is
	begin
		return ldu_base & imm; 
	end;
	
	
	function lds(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector is
	begin
		return lds_base & imm; 
	end;
	
	
	function rac(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector is
	begin
		return acc_base & '1' & rx; 
	end;
	
	
	function wac(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector is
	begin
		return acc_base & '0' & rx; 
	end;
		
	
	function add(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector is
	begin
		return add_base & '0' & rx; 
	end;
	
	
	function sub(rx : std_logic_vector(reg_wd - 1 downto 0)) return std_logic_vector is
	begin
		return add_base & '1' & rx; 
	end;

		
	function addi(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector is
	begin
		return adi_base & imm; 
	end;

	
	function ori(imm : std_logic_vector(bit_wd/2 - 1 downto 0)) return std_logic_vector is
	begin
		return ori_base & imm; 
	end;
	

end package body procdefs;
