library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

use work.procdefs.all;

entity execute is
	port (
		clk : in std_logic;
		rst : in std_logic;
		
		-- enables the alu
		alu_en : in std_logic;
		-- register file write enable
		rf_wr : in std_logic;
		
		-- sign extension
		exs : in std_logic;
		-- input selection for the alu: '0' use din, '1' use register defined by din
		ins : in std_logic;
		-- indicates whether to perform subtraction
		sub : in std_logic;
		-- result selector
		rss : in std_logic_vector(1 downto 0);
		
		-- data input for alu
		din : in std_logic_vector(bit_wd/2-1 downto 0);
		-- accumulator value
		acc : out std_logic_vector(bit_wd-1 downto 0)
	);
end entity execute;

architecture RTL of execute is
	
	component alu
		port(clk : in  std_logic;
			 rst : in  std_logic;
			 en  : in  std_logic;
			 rss : in  std_logic_vector(1 downto 0);
			 sub : in  std_logic;
			 din : in  std_logic_vector(bit_wd - 1 downto 0);
			 acc : out std_logic_vector(bit_wd - 1 downto 0));
	end component alu;
	
	component regfile
		port(clk  : in  std_logic;
			 rst  : in  std_logic;
			 wren : in  std_logic;
			 rd   : in  std_logic_vector(reg_wd - 1 downto 0);
			 din  : in  std_logic_vector(bit_wd - 1 downto 0);
			 dout : out std_logic_vector(bit_wd - 1 downto 0));
	end component regfile;
	
	signal alu_din, alu_dout, rf_din, rf_out : std_logic_vector(bit_wd-1 downto 0);
	signal ext : std_logic_vector(bit_wd/2-1 downto 0);
	
begin
	
	-- Sign extension
	ext     <= ... when ... else ...;
	
	-- Input selection
	alu_din <= ext & din when ins = '0' else rf_out;
	
	acc <= alu_dout;
	
	-- Alu instantiation
	alu_inst : alu
		port map(
			clk => clk,
			rst => rst,
			en  => alu_en,
			rss => rss,
			sub => sub,
			din => alu_din,
			acc => alu_dout
		);
		
	-- regfile instantiation
	rf_inst : regfile
		port map(
			clk  => clk,
			rst  => rst,
			wren => rf_wr,
			rd   => din(reg_wd-1 downto 0),
			din  => alu_dout,
			dout => rf_out
		);

end architecture RTL;
